/*
 * Include file for accessing the USB488 protocol stack API
 *
 * Contains function prototypes and useful defined constants for
 * calling USB488 protocol stack routines from a C/C++ application.
 */

#ifndef _USB488PS_H_
#define _USB488PS_H_

#include <windows.h>

typedef	void*	USBHANDLE;

typedef void ( CALLBACK* USB_ONSRQROUTINE )( USBHANDLE, WORD, LPARAM );

#define	DLLDECLSPEC_V		__declspec( dllimport ) void		__stdcall
#define	DLLDECLSPEC_W		__declspec( dllimport ) WORD		__stdcall
#define	DLLDECLSPEC_L		__declspec( dllimport ) long		__stdcall
#define	DLLDECLSPEC_I		__declspec( dllimport ) int			__stdcall
#define	DLLDECLSPEC_USB		__declspec( dllimport ) USBHANDLE	__stdcall
#define	DLLDECLSPEC_BOOL	__declspec( dllimport ) BOOL		__stdcall
#define	DLLDECLSPEC_DW		__declspec( dllimport ) DWORD		__stdcall

#ifdef __cplusplus
extern "C"{
#endif	//__cplusplus

DLLDECLSPEC_V		UsbInitialize( void );
DLLDECLSPEC_V		UsbFinalize( void );
DLLDECLSPEC_I		UsbGetDeviceList( int iNth, WORD *pBuffer, int pSize );
DLLDECLSPEC_USB		UsbOpenDevice( WORD w488ID );
DLLDECLSPEC_V		UsbCloseDevice( USBHANDLE hDev );
DLLDECLSPEC_W		UsbGetStatus( void );
DLLDECLSPEC_W		UsbGetErrorCode( void );
DLLDECLSPEC_L		UsbGetDataCount( void );
DLLDECLSPEC_V		UsbSetTimeout( USBHANDLE hDev, int iTime );
DLLDECLSPEC_I		UsbGetTimeout( USBHANDLE hDev );

//[Class Specific Request;Control In, Control Out]
DLLDECLSPEC_BOOL	UsbControlOUT( USBHANDLE hDev, UCHAR bmRequestType, UCHAR bRequest, USHORT wValue, USHORT wIndex, USHORT wLength, void* pDataStage );
DLLDECLSPEC_BOOL	UsbControlIN( USBHANDLE hDev, UCHAR bmRequestType, UCHAR bRequest, USHORT wValue, USHORT wIndex, USHORT wLength, void* pDataStage );
DLLDECLSPEC_V		UsbSetIFC( USBHANDLE hDev );									//Reset USB488 device??
DLLDECLSPEC_V		UsbDevClear( USBHANDLE hDev, USHORT wValue );					//Clear a single USB488 device
DLLDECLSPEC_V		UsbEnableRemote( USBHANDLE *pDevList );							//Enable remote USB488 programming for USB488 device
DLLDECLSPEC_V		UsbEnableLocal( USBHANDLE *pDevList );							//Enable operations from the front panel of USB488 devices
DLLDECLSPEC_V		UsbSendLLO( USBHANDLE *pDevList );								//Send the Local Lockout message to all USB488 devices
DLLDECLSPEC_V		UsbGetRemoteStatus( USBHANDLE hDev, BYTE *pStatus );			//Read remote status from USB488 device
DLLDECLSPEC_V		UsbTrigger( USBHANDLE hDev );									//Trigger a USB488 device
DLLDECLSPEC_V		UsbGetTriggerStatus( USBHANDLE hDev, BYTE *pStatus );			//Read Trigger status from USB488 device
DLLDECLSPEC_V		UsbSendIOCTL( USBHANDLE hDev, void* pDataStage, WORD wLength );	//I/O control??
DLLDECLSPEC_V		UsbClearStatusByte( USBHANDLE hDev );							//Clear status bytes of a single USB488 device
DLLDECLSPEC_V		UsbSet488ID( USBHANDLE hDev, WORD w488ID );						//Set USB488ID of USB488 device
DLLDECLSPEC_V		UsbGet488ID( USBHANDLE hDev, WORD *p488ID );					//Read USB488ID from USB488 device
DLLDECLSPEC_V		UsbGetCapacity( USBHANDLE hDev, void *pCapacity, WORD wLength );//Read USB488 device capacity

//[Bulk In, Bulk Out]
DLLDECLSPEC_V		UsbSend( USBHANDLE hDev, const void* pBuf, DWORD dwCount, DWORD dwMode );	//Bulk Out
DLLDECLSPEC_V		UsbReceive( USBHANDLE hDev, void* pBuf, DWORD dwCount, DWORD *pMode );		//Bulk In

DLLDECLSPEC_V		UsbCLS( USBHANDLE hDev );										//*CLS
DLLDECLSPEC_V		UsbSESE( USBHANDLE hDev, DWORD dwSESE );						//*ESE
DLLDECLSPEC_V		UsbSESEQuery( USBHANDLE hDev, DWORD *pResult );					//*ESE?
DLLDECLSPEC_V		UsbSESRQuery( USBHANDLE hDev, DWORD *pResult );					//*ESR?
DLLDECLSPEC_V		UsbIDNQuery( USBHANDLE hDev, char *pResult, int iBufSize );		//*IDN?
DLLDECLSPEC_V		UsbOPC( USBHANDLE hDev );										//*OPC
DLLDECLSPEC_V		UsbOPCQuery( USBHANDLE hDev, DWORD *pResult, DWORD dwTimeout );	//*OPC?
DLLDECLSPEC_V		UsbReset( USBHANDLE hDev);										//*RST
DLLDECLSPEC_V		UsbSRE( USBHANDLE hDev, DWORD dwData );							//*SRE
DLLDECLSPEC_V		UsbSREQuery( USBHANDLE hDev, DWORD *pResult );					//*SRE?
DLLDECLSPEC_V		UsbSTBQuery( USBHANDLE hDev, DWORD *pResult );					//*STB?
DLLDECLSPEC_V		UsbSelfTestQuery( USBHANDLE hDev, DWORD *pResult );				//*TST?
DLLDECLSPEC_V		UsbWait( USBHANDLE hDev );										//*WAI

DLLDECLSPEC_V		UsbSendBinaryWithCnt( USBHANDLE hDev, const void *pBuffer, DWORD dwCount, int iWidth );
DLLDECLSPEC_V		UsbSendBinary( USBHANDLE hDev, const void *pBuffer, DWORD dwCount, int iWidth );
DLLDECLSPEC_V		UsbSendBinaryWithHeader( USBHANDLE hDev, const char *pHeader, const void *pBuffer, DWORD dwCount, int iWidth );
DLLDECLSPEC_V		UsbSendBinaryData( USBHANDLE hDev, const char *pHeader, const void *pBuffer, DWORD dwCount, int iWidth, int iFuncID );
DLLDECLSPEC_V		UsbSendTextData( USBHANDLE hDev, const char *pBuffer );

DLLDECLSPEC_DW		UsbReceiveBinaryData( USBHANDLE hDev, char* pText, void* pData, DWORD dwCount, int* pWidth, int* pFuncID );
DLLDECLSPEC_DW		UsbReceiveTextData( USBHANDLE hDev, LPSTR pBuf, DWORD dwCount);

//[service request;Interrapt]
DLLDECLSPEC_V		UsbSetSrqRoutine( USBHANDLE hDev, USB_ONSRQROUTINE lpfnSrq,  LPARAM lParam );	//Interrapt in

#ifdef __cplusplus
}
#endif	//__cplusplus

// Return value of UsbGetStatus() function
#define	USB488_EERR	( 1 << 15 )	//Error
#define	USB488_TIMO	( 1 << 14 )	//Time limit exceeded
#define	USB488_SRQI	( 1 << 12 )	//SRQ line is asserted

// Return value of UsbErrorCode() function
#define	USB488_ENON	0	//Success
#define	USB488_EDVR	1	//System error
#define	USB488_ENOL	3	//No USB488Device on the USB
#define	USB488_EARG	5	//Invalid argument to function call
#define	USB488_EABO	7	//I/O operation aborted (timeout)
#define	USB488_ETAB	21	//Table problem

#define	USB488_EINI	128	//UsbInitialize is not executed.
#define	USB488_ECOM	129	//Commiunication error without timeout
#define	USB488_ECOL	130	//Collision with USB488ID
#define	USB488_EDAT	131	//Data commiunication error
#define	USB488_ESND	132	//BULK OUT ERROR. Another function execute without to send NULL packet.
#define	USB488_ERCV	133	//BULK IN ERROR. Another function execute without to receive NULL packet.
#define	USB488_EOPN	134	//Device is not available without UsbOpenDevice( WORD ) function.
#define	USB488_EHDL	135	//Invalid Usb488 Handle

// Class Request Type
#define	SET_CLASS_REQ	0x20
#define	GET_CLASS_REQ	0xA0

//Class Specific Request
#define	SET_IFC		0x01
#define	SET_SDC		0x02
#define	SET_REN		0x03
#define	GET_REN		0x04
#define	SET_TRG		0x05
#define	GET_TRG		0x06
#define	SET_IOCTL	0x07
#define	CLS_STB		0x08
#define	SET_488ID	0x09
#define	GET_488ID	0x0A
#define	GET_CAPA	0x0B

// Message ID
#define	MSG_CMD		0	// Common command
#define	MSG_BIN		1	// Binary data

// Function ID for Binary Data
#define	BIN_WITH_DC				0x01	// Binary data with data counts.
#define	BIN_WITHOUT_DC			0x02	// Binary data without data counts.
#define	BIN_WITHOUT_DC_WITH_HD	0x82	// Binary data with ASCII header without data counts.

// Eom Mode
#define	EOM_NULLPACKET	1
#define	EOM_CONTINUE	0

// Binary Width
#define	BINARY_BYTE		1
#define	BINARY_SHORT	2
#define	BINARY_LONG		4
#define	BINARY_SINGLE	4
#define	BINARY_DOUBLE	8

// Capacity Bits of UsbGetCapacity() function
#define	CAP_INTERRUPT_BIT			0x80	// Interupt transfer: 1=support
#define	CAP_488_BIT					0x10	// 0:488.1, 1:488.2
#define	CAP_ASCII_COMMON_CMD_BIT	0x08	// ASCII common command: 1=support
#define	CAP_ENDIAN_BIT				0x01	// 0:big, 1:little

// Request Type
#define	CONTROL_IN		0xA0
#define	CONTROL_OUT		0x20

// Return value of UsbGetRemoteStatus() function
#define	REM_STAT_LOCAL		0x00	// Local
#define	REM_STAT_REMOTE		0x01	// Remote
#define	REM_STAT_LOCKOUT	0x02	// Local Lockout

// Return value of UsbGetTriggerStatus() function
#define	TRIGGER_BUSY		0x00	// Busy
#define	TRIGGER_COMPLETE	0x01	// Complete

#endif
